<?php

/*
    Gère l'envoi et le stockage des fichiers de configuration ou de données associées.
        Le script est prévu pour être le plus générique possible afin de s'insérer dans tout type de site internet.
        Organisé sous forme de module, on ne définit ici que des fonctions afin de pouvoir être générique et aucune fonction d'authentification n'est incluse.

        Pour utiliser le module, il suffit d'appeler la fonction H6CfgUploadMain($_POST) après avoir contrôlé l'authentification.
*/


class Holo6ConfigManager {
    protected $dossier_racine_abs;
    protected $dossier_destination_defaut;
    protected $events_callback_name;
    protected $read_only_mode;

    public function __construct($events_callback_name=null, $dossier_racine_abs=null, $dossier_destination_defaut=null, $read_only_mode=false) {
        $this->events_callback_name = $events_callback_name;
        $this->read_only_mode = $read_only_mode;

        if (($dossier_racine_abs == null) || (strlen($dossier_racine_abs) == 0)) {
            $this->dossier_racine_abs = dirname(__FILE__);
        }
        else {
            $this->dossier_racine_abs = $dossier_racine_abs;
        }

        if (($dossier_destination_defaut == null) || (strlen($dossier_destination_defaut) == 0)) {
            $dossier_destination_defaut = "files";
        }
        $this->dossier_destination_defaut = $dossier_destination_defaut;
    }

    public function agent_config_file_exists($agent_role, $dossier_destination=null) /* -> bool */ {
        $nom_fichier_destination = $this->_get_config_filename_agent($agent_role);
        $dossier_destination_abs = $this->_get_dossier_destination_abs($dossier_destination);
        if ($dossier_destination_abs != null) {
            return file_exists($dossier_destination_abs . DIRECTORY_SEPARATOR . $nom_fichier_destination);
        }
        return false;
    }

    public function reset_default_agents_config($langue_iso2, $dossier_destination=null, $ignore_reset_if_exists=false) /* -> bool */ {
        /*
            Initialisation des fichiers de configuration pour la langue donnée en argument.
            Si la langue n'est pas trouvée, on utilisera une langue par défaut (typiquement "en").
            Le dossier de destination est relatif au dossier courant.
        */
        return ($this->reset_config_from_base($langue_iso2, "public", $dossier_destination, $ignore_reset_if_exists) &&
                $this->reset_config_from_base($langue_iso2, "writer", $dossier_destination, $ignore_reset_if_exists) &&
                $this->reset_config_from_base($langue_iso2, "summarizer", $dossier_destination, $ignore_reset_if_exists));
    }

    public function reset_config_from_base($langue_iso2, $agent_role, $dossier_destination=null, $ignore_reset_if_exists=false) /* -> bool */ {
        /*
            Initialisation du fichier de configuration par défaut pour le rôle en argument.
            Utilise le répertoire des fichiers de configuration de base.
            Si le fichier de configuration n'est pas trouvé pour la langue, on utilisera "en" en guise de valeur initiale.
            Le dossier de destination est relatif au dossier courant.
        */
        $dossier_racine_base_abs = $this->dossier_racine_abs . DIRECTORY_SEPARATOR . ".." . DIRECTORY_SEPARATOR . "base";
        $nom_fichier_base = "Holo6Config-base-{$langue_iso2}.json";
        if (! file_exists($dossier_racine_base_abs . DIRECTORY_SEPARATOR . $nom_fichier_base)) {
            $nom_fichier_base = "Holo6Config-base-en.json";
        }
        $chemin_reference_abs = $dossier_racine_base_abs . DIRECTORY_SEPARATOR . $nom_fichier_base;
        if (file_exists($chemin_reference_abs)) {
            // $this->_init_upload_dir($dossier_destination);
            $nom_fichier_destination = $this->_get_config_filename_agent($agent_role);
            $dossier_destination_rel = $this->_get_dossier_destination_relatif($dossier_destination);
            $chemin_fichier_rel_upload = $dossier_destination_rel . DIRECTORY_SEPARATOR . $nom_fichier_destination;
            // error_log("Fichier existe (" . $ignore_reset_if_exists . ") ? " . $dossier_destination_abs . DIRECTORY_SEPARATOR . $nom_fichier_destination);
            if ($this->_event_callback("init_config", $chemin_fichier_rel_upload, null)) {
                $json_file_data = file_get_contents($chemin_reference_abs);
                if ($json_file_data) {
                    $dict_data = json_decode($json_file_data, true);
                    if ($dict_data) {
                        if (isset($dict_data["core"]) && isset($dict_data["apps"])) {
                            if (isset($dict_data["core"]["client"]["roles"][$agent_role])) {
                                $dict_data["core"]["default_role"] = $agent_role;
                                if ($agent_role == "writer") {
                                    $dict_data["apps"]["ui"]["default_role"] = "writer";
                                    if (substr($dict_data["apps"]["ui"]["undocked_window_state"]["title"], -8) === " (Admin)") {
                                    }
                                    else {
                                        $dict_data["apps"]["ui"]["undocked_window_state"]["title"] .= " (Admin)";
                                    }
                                    $dict_data["apps"]["ui"]["style_params"]["themes"]["light"]["highlight_color_static"] = "#af4b4b";
                                    $dict_data["apps"]["ui"]["style_params"]["themes"]["light"]["highlight_color_hover"] = "#ff6068";
                                    // $dict_data["core"]["debug_level"] = 10;
                                }
                                else if ($agent_role == "summarizer") {
                                    $dict_data["core"]["debug_level"] = 10;
                                    $dict_data["apps"]["cl"]["default_role"] = "summarizer";
                                    if (isset($dict_data["apps"]["ui"])) {
                                        unset($dict_data["apps"]["ui"]);
                                    }
                                }
                                else {  // "public" typiquement...
                                    $dict_data["apps"]["ui"]["style_params"]["themes"]["light"]["highlight_color_static"] = "#58719d";
                                    $dict_data["apps"]["ui"]["style_params"]["themes"]["light"]["highlight_color_hover"] = "#218ebe";
                                }
                                $json_data = json_encode($dict_data, 0);
                                $resultat_ecriture = $this->write_agent_config($json_data, $agent_role, $dossier_destination, $ignore_reset_if_exists);
                                if (is_string($resultat_ecriture)) {  // C'est un message d'erreur...
                                    return false;
                                }
                                else {
                                    if ($resultat_ecriture === true || (is_numeric($resultat_ecriture) && ($resultat_ecriture > 0))) {
                                        return true;
                                    }
                                    else {
                                        return false;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
        else {
            error_log($this->_manage_error_response("[Holo6] Reference File not found : " . $chemin_reference_abs));
        }
        return false;
    }

    public function write_agent_config($json_data, $agent_role, $dossier_destination=null, $ignore_if_exists=false) /* -> bool|string|int */ {
        return $this->write_config_from_json($json_data, $this->_get_config_filename_agent($agent_role), $dossier_destination, $ignore_if_exists);
    }

    public function write_config_from_json($json_data, $nom_fichier_json, $dossier_destination=null, $ignore_if_exists=false) /* -> bool */ {
        /*
            Ecriture du fichier de configuration dont on fournit le contenu.
            Le dossier de destination est relatif au dossier courant.
        */
        $taille_max_fichier = 32; // En ko
        $dossier_destination_rel = $this->_get_dossier_destination_relatif($dossier_destination);
        if (self::holo6_config_check_filename($nom_fichier_json)) {
            if (is_string($json_data) && (strlen($json_data) <= ($taille_max_fichier * 1024))) {
                if ($json_data[0] === '{') {
                    if (self::holo6_config_check_filepath($dossier_destination_rel)) {
                        if ($this->_init_upload_dir($dossier_destination)) {
                            $chemin_fichier_rel_upload = $dossier_destination_rel . DIRECTORY_SEPARATOR . basename($nom_fichier_json);
                            if ($this->_event_callback("write_config", $chemin_fichier_rel_upload, $json_data)) {
                                if ($this->read_only_mode ) {
                                    return true;
                                }
                                $dossier_destination_abs = $this->_get_dossier_destination_abs($dossier_destination);
                                if ($dossier_destination_abs != null) {
                                    $chemin_abs_fichier_dest = $dossier_destination_abs . DIRECTORY_SEPARATOR . basename($nom_fichier_json);
                                    if ($ignore_if_exists && file_exists($chemin_abs_fichier_dest)) {
                                        return true;  /* On interrompt mais c'est pas une erreur... */
                                    }
                                    $resultat_ecriture = $this->_write_file($chemin_abs_fichier_dest, $json_data);
                                    if (($resultat_ecriture !== false) && ($resultat_ecriture === strlen($json_data))) {
                                        if (file_exists($chemin_abs_fichier_dest)) {
                                            return $resultat_ecriture;
                                        }
                                        else { return ($this->_manage_error_response(53, $dossier_destination_rel)); }
                                    }
                                    else { return ($this->_manage_error_response(54, $dossier_destination_rel)); }
                                }
                                else { return ($this->_manage_error_response(55, $dossier_destination_rel)); }
                            }
                            else { return ($this->_manage_error_response(56, $dossier_destination_rel)); }
                        }
                        else { return ($this->_manage_error_response(57, $dossier_destination_rel)); }
                    }
                    else { return ($this->_manage_error_response(58, $dossier_destination_rel)); }
                }
                else { return ($this->_manage_error_response(59, $dossier_destination_rel)); }
            }
            else { return ($this->_manage_error_response(60, $dossier_destination_rel)); }
        }
        else { return ($this->_manage_error_response(99, $dossier_destination_rel)); }
    }

    public function write_file_from_base64($data_base64, $nom_fichier, $dossier_destination=null, $ignore_if_exists=false) /* -> bool */ {
        $taille_max_fichier = 64; // En ko
        $dossier_destination_rel = $this->_get_dossier_destination_relatif($dossier_destination);
        if (self::holo6_config_check_filename($nom_fichier)) {
            if (is_string($data_base64) && (strlen($data_base64) <= ($taille_max_fichier * 1024))) {
                if ((substr($data_base64, 0, 9) === "base64://") && (substr($data_base64, 9, 5) === "iVBOR")) {
                    /* Entête PNG en base64.
                        Preuve : python```
                            ht = ht=bytes.fromhex('89') + b'PNG';
                            str(base64.b64encode(ht + bytes.fromhex('00'))) + " -> " + str(base64.b64encode(ht + bytes.fromhex('FF')))
                        ``` ->  */
                    if (self::holo6_config_check_filepath($dossier_destination_rel)) {
                        if ($this->_init_upload_dir($dossier_destination)) {
                            $chemin_fichier_rel_upload = $dossier_destination_rel . DIRECTORY_SEPARATOR . basename($nom_fichier);
                            $raw_data = base64_decode(substr($data_base64, 9, strlen($data_base64)-9), true);
                            if ($raw_data === false) {
                                return ($this->_manage_error_response(90, $dossier_destination_rel));
                            }
                            if ($this->_event_callback("write_file", $chemin_fichier_rel_upload, $raw_data)) {
                                if ($this->read_only_mode ) {
                                    return true;
                                }
                                $dossier_destination_abs = $this->_get_dossier_destination_abs($dossier_destination);
                                if ($dossier_destination_abs != null) {
                                    $chemin_abs_fichier_dest = $dossier_destination_abs . DIRECTORY_SEPARATOR . basename($nom_fichier);
                                    if ($ignore_if_exists && file_exists($chemin_abs_fichier_dest)) {
                                        return true;  /* On interrompt mais c'est pas une erreur... */
                                    }
                                    $resultat_ecriture = $this->_write_file($chemin_abs_fichier_dest, $raw_data);
                                    if ($resultat_ecriture !== false) {
                                        if (file_exists($chemin_abs_fichier_dest)) {
                                            return $resultat_ecriture;
                                        }
                                        else { return ($this->_manage_error_response(53, $dossier_destination_rel)); }
                                    }
                                    else { return ($this->_manage_error_response(54, $dossier_destination_rel)); }
                                }
                                else { return ($this->_manage_error_response(55, $dossier_destination_rel)); }
                            }
                            else { return ($this->_manage_error_response(56, $dossier_destination_rel)); }
                        }
                        else { return ($this->_manage_error_response(57, $dossier_destination_rel)); }
                    }
                    else { return ($this->_manage_error_response(58, $dossier_destination_rel)); }
                }
                else { return ($this->_manage_error_response(59, $dossier_destination_rel)); }
            }
            else { return ($this->_manage_error_response(60, $dossier_destination_rel)); }
        }
        else { return ($this->_manage_error_response(99, $dossier_destination_rel)); }
    }

    public static function holo6_config_check_filepath($dossier_destination_rel) /* -> bool */ {
        if ((substr($dossier_destination_rel, 0, 1) === '.') && (substr($dossier_destination_rel, 1, 1) !== DIRECTORY_SEPARATOR)) {
            /* Dans un dossier caché ?! */
            return false;
        }
        else {
            return true;
        }
        return false;
    }

    public static function holo6_config_check_filename($nom_fichier) /* -> bool */ {
        if (is_string($nom_fichier)) {
            if ((strlen($nom_fichier) < 128) && (strpos($nom_fichier, "..") === false)) {
                if (substr($nom_fichier, -5) === ".json") {
                    return (strlen($nom_fichier) > 17) && (substr($nom_fichier, -5) === ".json") && (substr($nom_fichier, 0, 12) === "Holo6Config-");
                }
                else if (substr($nom_fichier, -4) === ".png") {
                    return true;
                }
            }
        }
        return false;
    }

    public function manage_http_request($_post_data) /* -> void */ {
        /*
            Traites la requête via HTTP.
            _post_data contient les données envoyées par POST obtenues par json_decode(file_get_contents("php://input"), true)
            La fonction ne renvoi rien mais affiche le résultat via echo et assigne l'entête text/plain, ce qui signifie que l'appelant n'est pas censé ajouter de données au résultat renvoyé par le serveur.
        */
        header("Content-Type: text/plain");
        if (isset($_post_data["command"])) {
            $commande = $_post_data["command"];
            $resultat_commande = null;
            if ($commande === "get_session_token") {
                $resultat_commande = $this->_http_session_init();
            }
            else if ($commande === "init") {
                $resultat_commande = $this->_http_init_config($_post_data);
            }
            else if ($commande === "check_file") {
                $resultat_commande = $this->_http_check_file($_post_data);
            }
            else if ($commande === "get_file_content") {
                $resultat_commande = $this->_http_get_file_content($_post_data);
            }
            else if ($commande === "upload_file") {
                $resultat_commande = $this->_http_upload_file($_post_data);
            }
            else if ($commande === "remove_file") {
                $resultat_commande = $this->_http_remove_file($_post_data);
            }

            if (($resultat_commande != null) && (strlen($resultat_commande) > 0)) {
                echo($resultat_commande);
            }
        }
    }

    protected function _write_file($chemin_abs_fichier_dest, $data) /* -> bool|int */ {
        if ($this->read_only_mode ) {
            return true;
        }
        $resultat_ecriture = file_put_contents($chemin_abs_fichier_dest, $data);
        if (($resultat_ecriture !== false) && ($resultat_ecriture > 0)) {
            // chmod($chemin_abs_fichier_dest, 0664);
            chmod($chemin_abs_fichier_dest, 0644);
        }
        return $resultat_ecriture;
    }

    protected function _http_init_config($_post_data) /* -> str|null */ {
        /*
            Initialisation du fichier de configuration.
            Format de la requête avec la langue au format iso 2 (doit correspondre à un des ) : { "command": "init", "lang": "fr" } 
        */
        if (isset($_post_data["command"]) && ($_post_data["command"] === "init")) {
            if (isset($_post_data["lang"])) {
                $sub_dir = isset($_post_data["dir"]) ? $_post_data["dir"] : null;
                if ($this->reset_default_agents_config($_post_data["lang"], $sub_dir)) {
                    return "h6_config:ok";
                }
            }
        }
        return null;
    }

    protected function _http_get_file_content($_post_data) /* -> str|null */ {
        /*
            Renvoi le contenu du fichier de configuration (en texte).
            Format de la requête : { "command": "get_file_content", "file": { "name": "..." } } 
        */
        if (isset($_post_data["command"]) && ($_post_data["command"] === "get_file_content")) {
            if (isset($_post_data["file"])) {
                $nom_fichier = $_post_data["file"]["name"];
                $sub_dir = isset($_post_data["dir"]) ? $_post_data["dir"] : null;
                $dossier_destination_abs = $this->_get_dossier_destination_abs($sub_dir);
                if ($dossier_destination_abs != null) {
                    if (self::holo6_config_check_filename($nom_fichier)) {
                        $chemin_fichier_abs = $dossier_destination_abs . DIRECTORY_SEPARATOR . $nom_fichier;
                        if (file_exists($chemin_fichier_abs)) {
                            $contenu = file_get_contents($chemin_fichier_abs);
                            if ($contenu != false) {
                                return $contenu;
                            }
                        }
                    }
                }
            }
        }
        return null;
    }

    protected function _http_check_file($_post_data) /* -> str|null */ {
        /*
            Contrôle si le fichier de configuration existe.'
            Format de la requête : { "command": "check_file", "file": { "name": "..." } } 
        */
        if (isset($_post_data["command"]) && ($_post_data["command"] === "check_file")) {
            if (isset($_post_data["file"])) {
                $nom_fichier = $_post_data["file"]["name"];
                $sub_dir = isset($_post_data["dir"]) ? $_post_data["dir"] : null;
                $dossier_destination_abs = $this->_get_dossier_destination_abs($sub_dir);
                if ($dossier_destination_abs != null) {
                    if (self::holo6_config_check_filename($nom_fichier)) {
                        $chemin_fichier_abs = $dossier_destination_abs . DIRECTORY_SEPARATOR . $nom_fichier;
                        if (file_exists($chemin_fichier_abs)) {
                            return "h6_config:ok:" . $nom_fichier;
                        }
                    }
                }
            }
        }
        return null;
    }

    protected function _http_session_init() /* str|null */ {
        /*
            Clé de session interne à ce script. C'est en complément de X-CSRF-TOKEN fourni dans l'entête.
            Sera contrôlé lors de l'envoi d'un fichier (paramètre "h6_upload_token").
            Format de la requête : { "command": "get_session_token" } 
        */
        if (session_start()) {
            if (empty($_SESSION["h6_upload_token"])) {
                $_SESSION["h6_upload_token"] = bin2hex(random_bytes(32));
            }
            return $_SESSION["h6_upload_token"];
        }
        return null;
    }

    protected function _http_check_session($_post_data) {
        if (session_start()) {
            if ((!isset($_post_data["h6_upload_token"]) || !isset($_SESSION["h6_upload_token"])) || (!hash_equals($_SESSION["h6_upload_token"], $_post_data["h6_upload_token"]))) {
                return false;
            }
            return true;
        }
        return false;
    }

    protected function _http_remove_file($_post_data) /* -> str|null */ {
        if (isset($_post_data["command"]) && ($_post_data["command"] === "remove_file")) {
            if (!$this->_http_check_session($_post_data)) {
            }
            else if (isset($_post_data["file"]) && $_post_data["file"]["name"]) {
                $sub_dir = isset($_post_data["dir"]) ? $_post_data["dir"] : null;
                $dossier_destination_rel = $this->_get_dossier_destination_relatif($sub_dir);
                if ($dossier_destination_rel != null) {
                    $nom_fichier = $_post_data["file"]["name"];
                    if (substr($nom_fichier, -4) === ".png") {
                        $chemin_fichier_rel_upload = $dossier_destination_rel . DIRECTORY_SEPARATOR . $nom_fichier;
                        if ($this->_event_callback("remove_file", $chemin_fichier_rel_upload, null)) {
                            $dossier_destination_abs = $this->_get_dossier_destination_abs($sub_dir);
                            if ($dossier_destination_abs != null) {
                                $chemin_abs_fichier_dest = $dossier_destination_abs . DIRECTORY_SEPARATOR . basename($nom_fichier);
                                if (file_exists($chemin_abs_fichier_dest)) {
                                    if (unlink($chemin_abs_fichier_dest) === true) {
                                        return "h6_config:ok";
                                    }
                                    else {
                                        return ($this->_manage_error_response(5));
                                    }
                                }
                                else {
                                    return ($this->_manage_error_response(6));
                                }
                            }
                        }
                        else {
                            // return ($this->_manage_error_response(67));
                        }
                    }
                    else {
                        return ($this->_manage_error_response(68));
                    }
                }
                else {
                    return ($this->_manage_error_response(69));
                }
            }
            else {
                return ($this->_manage_error_response(70));
            }
        }
        return null;
    }

    protected function _http_upload_file($_post_data) /* -> str|null */ {
        /*
            Envoi un fichier.
            Format de la requête : { "command": "upload_file", "file": { "name": "...", "data": "..." }, "h6_upload_token": "..." } 
        */
        if (isset($_post_data["command"]) && ($_post_data["command"] === "upload_file")) {
            $ignore_if_exists = false;
            if (!$this->_http_check_session($_post_data)) {
                return ($this->_manage_error_response(3));
            }
            else if (isset($_post_data["file"]) && $_post_data["file"]["name"]) {
                $sub_dir = isset($_post_data["dir"]) ? $_post_data["dir"] : null;
                $dossier_destination_rel = $this->_get_dossier_destination_relatif($sub_dir);
                if ($dossier_destination_rel != null) {
                    $nom_fichier = $_post_data["file"]["name"];
                    $chemin_fichier_rel_upload = $dossier_destination_rel . DIRECTORY_SEPARATOR . $nom_fichier;
                    if (substr($nom_fichier, -5) === ".json") {
                        if ($this->_event_callback("recv_config", $chemin_fichier_rel_upload, null)) {
                            $fileContent = $_post_data["file"]["data"];
                            $resultat_ecriture = $this->write_config_from_json($fileContent, $nom_fichier, $sub_dir, $ignore_if_exists);
                            if (is_string($resultat_ecriture)) {  // C'est un message d'erreur...
                                return $resultat_ecriture;
                            }
                            else {
                                if ($resultat_ecriture === true || (is_numeric($resultat_ecriture) && ($resultat_ecriture > 0))) {
                                    return "h6_config:ok:" . $nom_fichier;
                                }
                                else {
                                    return ($this->_manage_error_response(5, strlen($fileContent) . " -> " . strval($resultat_ecriture)));
                                }
                            }
                        }
                        else {
                            return ($this->_manage_error_response(67));
                        }
                    }
                    else if (substr($nom_fichier, -4) === ".png") {
                        if ($this->_event_callback("recv_file", $chemin_fichier_rel_upload, null)) {
                            $fileContent = $_post_data["file"]["data"];
                            $resultat_ecriture = $this->write_file_from_base64($fileContent, $nom_fichier, $sub_dir, $ignore_if_exists);
                            if (is_string($resultat_ecriture)) {  // C'est un message d'erreur...
                                return $resultat_ecriture;
                            }
                            else {
                                if ($resultat_ecriture === true || (is_numeric($resultat_ecriture) && ($resultat_ecriture > 0))) {
                                    return "h6_config:ok:" . $nom_fichier;
                                }
                                else {
                                    return ($this->_manage_error_response(5, strlen($fileContent) . " -> " . strval($resultat_ecriture)));
                                }
                            }
                        }
                        else {
                            return ($this->_manage_error_response(67));
                        }
                    }
                    else {
                        return ($this->_manage_error_response(68));
                    }
                }
                else {
                    return ($this->_manage_error_response(69));
                }
            }
            else {
                return ($this->_manage_error_response(70));
            }
            // session_unset();
            // session_destroy();
        }
        return null;
    }

    protected function _get_config_filename_agent($agent_role) {
        // return "Holo6Config-" . $agent_role . "@" . $_SERVER["SERVER_NAME"] . ".json";
        return "Holo6Config-" . $agent_role . ".json";
    }

    protected function _get_dossier_destination_relatif($sub_dir_uploads=null) {  /* Dossier de destination des fichiers (relatif) */
        if (($sub_dir_uploads != null) && (strlen($sub_dir_uploads) > 0)) {
            if (($sub_dir_uploads[0] == "/") || (strpos($sub_dir_uploads, "..") !== false)) {
                /* Interdit tout ce qui n'est pas un sous-dossier de la racine. Intedit les chemins absolus et permettant de remonter au parent. */
                return null;
            }
            else {
                return $sub_dir_uploads;
            }
        }
        return $this->dossier_destination_defaut;
    }

    protected function _init_upload_dir($sub_dir_uploads=null) /* bool */ {  /* On s'assure que le répertoire recevant les fichiers de configuration existe. */
        if ($this->read_only_mode) {
            return true;
        }
        else {
            $dossier_destination_abs = $this->_get_dossier_destination_abs($sub_dir_uploads);
            if ($dossier_destination_abs != null) {
                if ($this->_get_dossier_destination_relatif($sub_dir_uploads) === '.') {  // Dossier courant ?!
                }
                else if (!is_dir($dossier_destination_abs)) { // Création du dossier contenant les configurations si nécessaire.
                    if (mkdir($dossier_destination_abs, 0755, true)) {
                        $this->_write_file($dossier_destination_abs . DIRECTORY_SEPARATOR . "index.html", "");
                    }
                }
                return is_dir($dossier_destination_abs) && is_writable($dossier_destination_abs);
            }
        }
        return false;
    }

    protected function _get_dossier_destination_abs($sub_dir) {  /* Dossier de destination des fichiers (absolu) */
        $dossier_destination_rel = $this->_get_dossier_destination_relatif($sub_dir);
        if ($dossier_destination_rel != null) {
            return $this->dossier_racine_abs . DIRECTORY_SEPARATOR . $dossier_destination_rel;
        }
        return null;
    }

    protected function _manage_error_response($error_id, $extra_message=null) /* -> str */ {
        // die("Holo6ConfigManager Rejected. Error id : {$error_id}" . ($extra_message ? (" -> " . $extra_message) : ""));  // DEVELOPPEMENT, permet d'obtenir l'erreur dans le navigateur.
        return ("error:Holo6ConfigManager Rejected. Error id : {$error_id} (" . ($extra_message ? $extra_message : "null") .")");
    }

    protected function _event_callback($event_type_str, $file_path, $extra_data=null) {
        if (($this->events_callback_name != null) && (strlen($this->events_callback_name) > 0)) {
            $r = call_user_func($this->events_callback_name, $event_type_str, $file_path, $extra_data);
            if (isset($r)) {
                return $r;
            }
        }
        return true;  // Comportement par défaut, si la fonction n'est pas définie, on ne déclenche aucun refus.
    }
}


function H6CfgUploadMain($_post_data=null, $events_callback_name=null, $dossier_racine_abs=null, $dossier_destination_defaut=null, $read_only_mode=false) {
    if ($_post_data == null) {
        if ($_SERVER["REQUEST_METHOD"] == "POST") {
            // $_post_data = $_POST; // TEST
            $_post_data = json_decode(file_get_contents("php://input"), true);
        }
    }
    else {
        $_post_data = json_decode($_post_data, true);
    }
    if ($_post_data != null) {
        $uploads_instance = new Holo6ConfigManager($events_callback_name, $dossier_racine_abs, $dossier_destination_defaut, $read_only_mode);
        $uploads_instance->manage_http_request($_post_data);
    }
}

?>
